
#!/usr/bin/env python3
"""
Daily digest builder (HTML skeleton) for #OpTanzania
- Reads latest JSONL files and verified media CSV
- Renders a minimal HTML report using Jinja2
"""
import os, json, argparse, glob, datetime
from jinja2 import Template

TEMPLATE = """<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>#OpTanzania Daily Digest - {{ date }}</title>
<style>
 body { font-family: system-ui, Segoe UI, Roboto, Arial, sans-serif; background:#0c0f10; color:#e6f5e6; }
 h1, h2 { color:#6cff6c; }
 a { color:#9af59a; }
 .card { background:#121618; border:1px solid #1f2a22; border-radius:10px; padding:16px; margin:12px 0; }
 .grid { display:grid; grid-template-columns: repeat(auto-fit, minmax(260px, 1fr)); gap:12px; }
 .muted { color:#9fb6a8; font-size: 0.95em; }
 code { background:#0b120f; padding:2px 6px; border-radius:6px; }
</style>
</head>
<body>
<h1>#OpTanzania — Daily Digest</h1>
<p class="muted">Date: {{ date }} • Generated locally</p>

<h2>News/Official</h2>
<div class="grid">
{% for rec in news[:50] %}
  <div class="card">
    <strong>{{ rec.title or "Untitled" }}</strong><br>
    <a href="{{ rec.link }}" target="_blank">{{ rec.link }}</a><br>
    {% if rec.published %}<span class="muted">{{ rec.published }}</span>{% endif %}
  </div>
{% endfor %}
</div>

<h2>Blackout Monitors</h2>
<div class="grid">
{% for rec in blackouts[:30] %}
  <div class="card">
    <strong>{{ rec.title or "Network Event" }}</strong><br>
    <a href="{{ rec.link }}" target="_blank">{{ rec.link }}</a><br>
    {% if rec.discovered_at %}<span class="muted">{{ rec.discovered_at }}</span>{% endif %}
  </div>
{% endfor %}
</div>

<h2>Social (Recent API Captures)</h2>
<div class="grid">
{% for rec in social[:20] %}
  <div class="card">
    <pre style="white-space: pre-wrap;">{{ rec | tojson(indent=2) }}</pre>
  </div>
{% endfor %}
</div>

</body>
</html>"""

def read_jsonl(path):
    out = []
    if not os.path.exists(path): return out
    with open(path, "r", encoding="utf-8") as f:
        for line in f:
            try:
                out.append(json.loads(line))
            except Exception:
                continue
    return out

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--date", default=str(datetime.date.today()))
    ap.add_argument("--out", default="reports/daily_digest/digest.html")
    args = ap.parse_args()

    news = read_jsonl("raw_data/news/news.jsonl")
    blackouts = read_jsonl("raw_data/blackouts/blackouts.jsonl")
    social = read_jsonl("raw_data/social/social.jsonl")

    tpl = Template(TEMPLATE)
    html = tpl.render(date=args.date, news=news, blackouts=blackouts, social=social)

    os.makedirs(os.path.dirname(args.out), exist_ok=True)
    with open(args.out, "w", encoding="utf-8") as f:
        f.write(html)

    print(f"[digest] Wrote {args.out}")

if __name__ == "__main__":
    main()
